#pragma once
#include "EnginePtr.h"

namespace storm {
	STORM_PKG(core);

	/**
	 * Class representing a single character as defined by the Str class. Currently one character is
	 * equal to one unicode code point.
	 */
	class Char {
		STORM_VALUE;
	public:
		// Nice to have from C++.
		Char(char ch);
		Char(wchar ch);
		Char(nat16 ch);

#ifdef POSIX
		Char(wchar_t ch);
#endif

		// Create a character from a codepoint number.
		STORM_CTOR Char(Nat codepoint);

		// Create the null character.
		STORM_CTOR Char();

		// Compare.
		Bool STORM_FN operator ==(Char o) const;
		Bool STORM_FN operator !=(Char o) const;

		// Hash.
		Nat STORM_FN hash() const;

		// Deep copy.
		void STORM_FN deepCopy(CloneEnv *env);

		/**
		 * C++ interface. May change if we change encoding.
		 */

		// Leading/trailing surrogate pair. Leading may be 0, which means we fit into one wchar.
		wchar leading() const;
		wchar trailing() const;

		// # of codepoints in use.
		nat size() const;

		// Get the codepoint.
		inline Nat STORM_FN codepoint() const { return value; }

		// Get the leading UTF-16 codepoint. If it is zero, then this char is only one codepoint.
		inline Nat STORM_FN utf16Leading() const { return Nat(leading()); }

		// Get the trailing UTF-16 codepoint. It is always valid.
		inline Nat STORM_FN utf16Trailing() const { return Nat(trailing()); }

		// Check if a code unit is an UTF-16 leading surrogate pair.
		static Bool STORM_FN utf16IsLeading(Nat unit);

		// Check if a code unit is an UTF-16 trailing surrogate pair.
		static Bool STORM_FN utf16IsTrailing(Nat unit);

		// Assemble a char from two UTF-16 code units. The leading codepoint should be 0 if only one
		// codepoint is present. If only a trailing codepoint is provided, or the encoding is
		// otherwise invalid, Char(0) is returned.
		static Char STORM_FN utf16Assemble(Nat leading, Nat trailing);

		// Output.
		void STORM_FN toS(StrBuf *to) const;

	private:
		Nat value;
	};

	// Output.
	Str *STORM_FN toS(EnginePtr e, Char ch);
	wostream &operator <<(wostream &to, const Char &ch);

}
