//    \\ SPIKE: Secure your secrets with SPIFFE. — https://spike.ist/
//  \\\\\ Copyright 2024-present SPIKE contributors.
// \\\\\\\ SPDX-License-Identifier: Apache-2.0

package secret

import (
	"testing"

	"github.com/spf13/cobra"
	"github.com/spiffe/go-spiffe/v2/workloadapi"
)

func TestNewSecretUndeleteCommand(t *testing.T) {
	source := &workloadapi.X509Source{}
	SPIFFEIDPattern := "^spiffe://example\\.org/spike$"

	cmd := newSecretUndeleteCommand(source, SPIFFEIDPattern)

	if cmd == nil {
		t.Fatal("Expected command to be created, got nil")
		return
	}

	if cmd.Use != "undelete <path>" {
		t.Errorf("Expected command use to be 'undelete <path>', got '%s'",
			cmd.Use)
	}

	if cmd.Short != "Undelete secrets at the specified path" {
		t.Errorf("Expected command short description to be "+
			"'Undelete secrets at the specified path', got '%s'", cmd.Short)
	}

	// Check if the `versions` flag is present
	flag := cmd.Flags().Lookup("versions")
	if flag == nil {
		t.Error("Expected flag 'versions' to be present")
	}
	if flag != nil && flag.Shorthand != "v" {
		t.Errorf("Expected flag shorthand to be 'v', got '%s'", flag.Shorthand)
	}
}

func TestNewSecretUndeleteCommandWithNilSource(t *testing.T) {
	SPIFFEIDPattern := "^spiffe://example\\.org/spike$"

	cmd := newSecretUndeleteCommand(nil, SPIFFEIDPattern)

	if cmd == nil {
		t.Fatal("Expected command to be created even with nil source, got nil")
		return
	}

	// Command should still be created; the nil source is handled at runtime
	if cmd.Use != "undelete <path>" {
		t.Errorf("Expected command use to be 'undelete <path>', got '%s'",
			cmd.Use)
	}
}

// TestSecretUndeleteCommandRegistration tests that the undelete command is
// registered properly as a subcommand of the secret command.
func TestSecretUndeleteCommandRegistration(t *testing.T) {
	source := &workloadapi.X509Source{}
	SPIFFEIDPattern := "^spiffe://example\\.org/spike$"

	secretCmd := NewCommand(source, SPIFFEIDPattern)

	var undeleteCmd *cobra.Command
	for _, cmd := range secretCmd.Commands() {
		if cmd.Use == "undelete <path>" {
			undeleteCmd = cmd
			break
		}
	}

	if undeleteCmd == nil {
		t.Error("Expected 'undelete' command to be registered")
	}
}

func TestSecretUndeleteCommandVersionsFlagDefault(t *testing.T) {
	source := &workloadapi.X509Source{}
	SPIFFEIDPattern := "^spiffe://example\\.org/spike$"

	cmd := newSecretUndeleteCommand(source, SPIFFEIDPattern)

	flag := cmd.Flags().Lookup("versions")
	if flag == nil {
		t.Fatal("Expected 'versions' flag to be present")
		return
	}

	if flag.DefValue != "0" {
		t.Errorf("Expected default value to be '0', got '%s'", flag.DefValue)
	}
}
