import unittest, operator

from solve import scalar, vector, matrix, identitymatrix, trafo, identitytrafo, Solver

class ScalarTestCase(unittest.TestCase):

    def testInit(self):
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), scalar, "")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), scalar, 1j)
        self.failUnlessEqual(str(scalar()), "unnamed_scalar")
        self.failUnlessEqual(str(scalar(name="s")), "s")
        self.failUnlessEqual(str(scalar(1)), "unnamed_scalar{=1.0}")
        self.failUnlessEqual(str(scalar(-1, name="s")), "s{=-1.0}")

    def testMath(self):
        self.failUnlessEqual(str(-scalar(name="s")), "unnamed_scalar{=-1.0} * s")
        self.failUnlessEqual(str(scalar(name="s") + scalar(name="t")), "s  +  t")
        self.failUnlessEqual(str(scalar(name="s") + 1), "s  +  unnamed_scalar{=1.0}")
        self.failUnlessEqual(str(1 + scalar(name="s")), "s  +  unnamed_scalar{=1.0}")
        self.failUnlessEqual(str(scalar(name="s") - scalar(name="t")), "unnamed_scalar{=-1.0} * t  +  s")
        self.failUnlessEqual(str(1 - scalar(name="s")), "unnamed_scalar{=-1.0} * s  +  unnamed_scalar{=1.0}")
        self.failUnlessEqual(str(2 * scalar(name="s")), "s * unnamed_scalar{=2.0}")
        self.failUnlessEqual(str(scalar(name="s") * 2), "s * unnamed_scalar{=2.0}")
        self.failUnlessEqual(str(scalar(name="s") * scalar(name="t")), "s * t")
        self.failUnlessEqual(str((scalar(name="s") + scalar(name="t")) * 2), "s * unnamed_scalar{=2.0}  +  t * unnamed_scalar{=2.0}")
        self.failUnlessEqual(str(scalar(name="s") / 2.0), "unnamed_scalar{=0.5} * s")
        self.failUnlessEqual(str(scalar(name="s") / 2), "unnamed_scalar{=0.0} * s") # integer logic!
        self.failUnlessEqual(str((scalar(name="s") + scalar(name="t")) / 2.0), "unnamed_scalar{=0.5} * s  +  unnamed_scalar{=0.5} * t")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, 2, scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, scalar(), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, vector(1), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, scalar() + scalar(), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, vector(1) + vector(1), scalar())

    def testAccess(self):
        s = scalar()
        self.failUnlessEqual(s.is_set(), 0)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), s.get)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), float, s)
        s.set(2)
        self.failUnlessEqual(s.is_set(), 1)
        self.failUnlessAlmostEqual(s.get(), 2.0)
        self.failUnlessAlmostEqual(float(s), 2.0)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), s.set, 3)
        self.failUnlessEqual(s.is_set(), 1)
        self.failUnlessAlmostEqual(s.get(), 2.0)
        self.failUnlessAlmostEqual(float(s), 2.0)

        s = scalar(2)
        self.failUnlessEqual(scalar(2).is_set(), 1)
        self.failUnlessAlmostEqual(s.get(), 2.0)
        self.failUnlessAlmostEqual(float(s), 2.0)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), s.set, 3)
        self.failUnlessEqual(s.is_set(), 1)
        self.failUnlessAlmostEqual(s.get(), 2.0)
        self.failUnlessAlmostEqual(float(s), 2.0)


class VectorTestCase(unittest.TestCase):

    def testInit(self):
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), vector, 0, 0)
        self.failUnlessEqual(str(vector(2)), "unnamed_vector{=(unnamed_vector[0], unnamed_vector[1])}")
        self.failUnlessEqual(str(vector([1, 2])), "unnamed_vector{=(unnamed_vector[0]{=1.0}, unnamed_vector[1]{=2.0})}")
        self.failUnlessEqual(str(vector(3, "a")), "a{=(a[0], a[1], a[2])}")
        self.failUnlessEqual(str(vector([3, 2, 1], "a")), "a{=(a[0]{=3.0}, a[1]{=2.0}, a[2]{=1.0})}")

    def testAccess(self):
        a = vector(2)
        self.failUnlessEqual(str(a), "unnamed_vector{=(unnamed_vector[0], unnamed_vector[1])}")
        self.failUnlessEqual(a[0].is_set(), 0)
        self.failUnlessEqual(a.x.is_set(), 0)
        self.failUnlessEqual(a[1].is_set(), 0)
        self.failUnlessEqual(a.y.is_set(), 0)
        self.failUnlessRaises(IndexError, operator.__getitem__, a, 2)
        self.failUnlessRaises(IndexError, getattr, a, "z")

        a[0].set(2)
        self.failUnlessEqual(str(a), "unnamed_vector{=(unnamed_vector[0]{=2.0}, unnamed_vector[1])}")
        self.failUnlessAlmostEqual(a[0].get(), 2.0)
        self.failUnlessAlmostEqual(float(a[0]), 2.0)
        self.failUnlessAlmostEqual(a.x.get(), 2.0)
        self.failUnlessAlmostEqual(float(a.x), 2.0)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), a[1].get)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), float, a[1])
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), a.y.get)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), float, a.y)
        self.failUnlessEqual(a[0].is_set(), 1)
        self.failUnlessEqual(a.x.is_set(), 1)
        self.failUnlessEqual(a[1].is_set(), 0)
        self.failUnlessEqual(a.y.is_set(), 0)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), a[0].set, 3)
        self.failUnlessAlmostEqual(a[0].get(), 2.0)
        self.failUnlessAlmostEqual(float(a[0]), 2.0)
        self.failUnlessAlmostEqual(a.x.get(), 2.0)
        self.failUnlessAlmostEqual(float(a.x), 2.0)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), a[1].get)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), float, a[1])
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), a.y.get)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), float, a.y)
        self.failUnlessEqual(a[0].is_set(), 1)
        self.failUnlessEqual(a.x.is_set(), 1)
        self.failUnlessEqual(a[1].is_set(), 0)
        self.failUnlessEqual(a.y.is_set(), 0)

        a[1].set(3)
        self.failUnlessEqual(str(a), "unnamed_vector{=(unnamed_vector[0]{=2.0}, unnamed_vector[1]{=3.0})}")
        self.failUnlessAlmostEqual(a[0].get(), 2.0)
        self.failUnlessAlmostEqual(float(a[0]), 2.0)
        self.failUnlessAlmostEqual(a.x.get(), 2.0)
        self.failUnlessAlmostEqual(float(a.x), 2.0)
        self.failUnlessAlmostEqual(a[1].get(), 3.0)
        self.failUnlessAlmostEqual(float(a[1]), 3.0)
        self.failUnlessAlmostEqual(a.y.get(), 3.0)
        self.failUnlessAlmostEqual(float(a.y), 3.0)
        self.failUnlessEqual(a[0].is_set(), 1)
        self.failUnlessEqual(a.x.is_set(), 1)
        self.failUnlessEqual(a[1].is_set(), 1)
        self.failUnlessEqual(a.y.is_set(), 1)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), a[0].set, 4)
        self.failUnlessAlmostEqual(a[0].get(), 2.0)
        self.failUnlessAlmostEqual(float(a[0]), 2.0)
        self.failUnlessAlmostEqual(a.x.get(), 2.0)
        self.failUnlessAlmostEqual(float(a.x), 2.0)
        self.failUnlessAlmostEqual(a[1].get(), 3.0)
        self.failUnlessAlmostEqual(float(a[1]), 3.0)
        self.failUnlessAlmostEqual(a.y.get(), 3.0)
        self.failUnlessAlmostEqual(float(a.y), 3.0)
        self.failUnlessEqual(a[0].is_set(), 1)
        self.failUnlessEqual(a.x.is_set(), 1)
        self.failUnlessEqual(a[1].is_set(), 1)
        self.failUnlessEqual(a.y.is_set(), 1)

        a = vector([1, 2, 3])
        self.failUnlessEqual(str(a.x), "unnamed_vector[0]{=1.0}")
        self.failUnlessEqual(str(a.y), "unnamed_vector[1]{=2.0}")
        self.failUnlessEqual(str(a.z), "unnamed_vector[2]{=3.0}")

    def testLen(self):
        for i in range(1, 5):
            a = vector(i)
            self.failUnlessEqual(len(a), i)
            self.failUnlessEqual(str(a), "unnamed_vector{=(" + ", ".join(["unnamed_vector[%i]" % j for j in range(i)]) + ")}")

    def testMath(self):
        self.failUnlessEqual(str(-vector(2, "a")), "unnamed_vector{=(unnamed_scalar{=-1.0} * a[0], unnamed_scalar{=-1.0} * a[1])}")
        self.failUnlessEqual(str(vector(2, "a") + vector(2, "b")), "unnamed_vector{=(a[0]  +  b[0], a[1]  +  b[1])}")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, 1, vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, vector(2), 1)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, scalar(), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, vector(2), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, scalar() + scalar(), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, vector(2), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, vector(2), vector(3))
        self.failUnlessEqual(str(vector(2, "a") - vector(2, "b")), "unnamed_vector{=(unnamed_scalar{=-1.0} * b[0]  +  a[0], unnamed_scalar{=-1.0} * b[1]  +  a[1])}")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, 1, vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, vector(2), 1)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, scalar(), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, vector(2), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, scalar() + scalar(), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, vector(2), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, vector(2), vector(3))
        self.failUnlessEqual(str(2 * vector(2, "a")), "unnamed_vector{=(a[0] * unnamed_scalar{=2.0}, a[1] * unnamed_scalar{=2.0})}")
        self.failUnlessEqual(str(vector(2, "a") * 2), "unnamed_vector{=(a[0] * unnamed_scalar{=2.0}, a[1] * unnamed_scalar{=2.0})}")
        self.failUnlessEqual(str(scalar(name="s") * vector(2, "a")), "unnamed_vector{=(a[0] * s, a[1] * s)}")
        self.failUnlessEqual(str(scalar(name="s") * (vector(2, "a") + vector(2, "b"))), "unnamed_vector{=(a[0] * s  +  b[0] * s, a[1] * s  +  b[1] * s)}")
        self.failUnlessEqual(str((scalar(name="s") + scalar(name="t")) * vector(2, "a")), "unnamed_vector{=(a[0] * s  +  a[0] * t, a[1] * s  +  a[1] * t)}")
        self.failUnlessEqual(str((scalar(name="s") + scalar(name="t")) * (vector(2, "a") + vector(2, "b"))), "unnamed_vector{=(a[0] * s  +  b[0] * s  +  a[0] * t  +  b[0] * t, a[1] * s  +  b[1] * s  +  a[1] * t  +  b[1] * t)}")
        self.failUnlessEqual(str(vector(2, "a") * scalar(name="s")), "unnamed_vector{=(a[0] * s, a[1] * s)}")
        self.failUnlessEqual(str(vector(2, "a") * vector(2, "b")), "a[0] * b[0]  +  a[1] * b[1]")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__mul__, vector(2, "a"), vector(3))
        self.failUnlessEqual(str(vector(2, "a") / 2.0), "unnamed_vector{=(unnamed_scalar{=0.5} * a[0], unnamed_scalar{=0.5} * a[1])}")
        self.failUnlessEqual(str(vector(2, "a") / 2), "unnamed_vector{=(unnamed_scalar{=0.0} * a[0], unnamed_scalar{=0.0} * a[1])}") # integer logic!
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, scalar(), vector(1))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, scalar() + scalar(), vector(1))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, vector(1), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, vector(1), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, vector(1), vector(1))


class MatrixTestCase(unittest.TestCase):

    def testInit(self):
        self.failUnlessEqual(str(matrix((2, 3))), "unnamed_matrix{=((unnamed_matrix[0, 0], unnamed_matrix[0, 1], unnamed_matrix[0, 2]), (unnamed_matrix[1, 0], unnamed_matrix[1, 1], unnamed_matrix[1, 2]))}")
        self.failUnlessEqual(str(matrix([[1, 2, 3], [4, 5, 6]])), "unnamed_matrix{=((unnamed_matrix[0, 0]{=1.0}, unnamed_matrix[0, 1]{=2.0}, unnamed_matrix[0, 2]{=3.0}), (unnamed_matrix[1, 0]{=4.0}, unnamed_matrix[1, 1]{=5.0}, unnamed_matrix[1, 2]{=6.0}))}")
        self.failUnlessEqual(str(matrix((2, 3), "a")), "a{=((a[0, 0], a[0, 1], a[0, 2]), (a[1, 0], a[1, 1], a[1, 2]))}")
        self.failUnlessEqual(str(matrix([[1, 2, 3], [4, 5, 6]], "a")), "a{=((a[0, 0]{=1.0}, a[0, 1]{=2.0}, a[0, 2]{=3.0}), (a[1, 0]{=4.0}, a[1, 1]{=5.0}, a[1, 2]{=6.0}))}")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), matrix, [[1, 2, 3], [4, 5]])

    def testLen(self):
        for rows in range(1, 5):
            for cols in range(1, 5):
                A = matrix((rows, cols))
                self.failUnlessEqual(A.getnumberofrows(), rows)
                self.failUnlessEqual(A.getnumberofcols(), cols)
                self.failUnlessEqual(str(A), "unnamed_matrix{=(" + ", ".join(["(" + ", ".join(["unnamed_matrix[%i, %i]" % (i, j) for j in range(cols)]) + ")" for i in range(rows)]) + ")}")

    def testAccess(self):
        A = matrix([2, 3])
        self.failUnlessEqual(str(A), "unnamed_matrix{=((unnamed_matrix[0, 0], unnamed_matrix[0, 1], unnamed_matrix[0, 2]), (unnamed_matrix[1, 0], unnamed_matrix[1, 1], unnamed_matrix[1, 2]))}")
        self.failUnlessEqual(A[0, 0].is_set(), 0)
        self.failUnlessEqual(A[1, 0].is_set(), 0)
        self.failUnlessRaises(IndexError, operator.__getitem__, A, (2, 0))
        self.failUnlessEqual(A[0, 1].is_set(), 0)
        self.failUnlessEqual(A[1, 1].is_set(), 0)
        self.failUnlessEqual(A[0, 2].is_set(), 0)
        self.failUnlessEqual(A[1, 2].is_set(), 0)
        self.failUnlessRaises(IndexError, operator.__getitem__, A, (0, 3))
        A[0, 0].set(1)
        self.failUnlessEqual(A[0, 0].is_set(), 1)
        self.failUnlessEqual(A[0, 1].is_set(), 0)
        self.failUnlessEqual(A[1, 0].is_set(), 0)
        self.failUnlessAlmostEqual(A[0, 0].get(), 1.0)
        self.failUnlessAlmostEqual(float(A[0, 0]), 1.0)

    def testMath(self):
        self.failUnlessEqual(str(-matrix([2, 3], "A")), "unnamed_matrix{=((unnamed_scalar{=-1.0} * A[0, 0], unnamed_scalar{=-1.0} * A[0, 1], unnamed_scalar{=-1.0} * A[0, 2]), (unnamed_scalar{=-1.0} * A[1, 0], unnamed_scalar{=-1.0} * A[1, 1], unnamed_scalar{=-1.0} * A[1, 2]))}")
        self.failUnlessEqual(str(matrix([2, 3], "A") + matrix([2, 3], "B")), "unnamed_matrix{=((A[0, 0]  +  B[0, 0], A[0, 1]  +  B[0, 1], A[0, 2]  +  B[0, 2]), (A[1, 0]  +  B[1, 0], A[1, 1]  +  B[1, 1], A[1, 2]  +  B[1, 2]))}")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, matrix([2, 3]), 1)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, matrix([2, 3]), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, matrix([2, 3]), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, matrix([2, 3]), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, matrix([2, 3]), vector(3))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, matrix([2, 3]), matrix([2, 4]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, matrix([2, 3]), matrix([3, 3]))
        self.failUnlessEqual(str(matrix([2, 3], "A") - matrix([2, 3], "B")), "unnamed_matrix{=((unnamed_scalar{=-1.0} * B[0, 0]  +  A[0, 0], unnamed_scalar{=-1.0} * B[0, 1]  +  A[0, 1], unnamed_scalar{=-1.0} * B[0, 2]  +  A[0, 2]), (unnamed_scalar{=-1.0} * B[1, 0]  +  A[1, 0], unnamed_scalar{=-1.0} * B[1, 1]  +  A[1, 1], unnamed_scalar{=-1.0} * B[1, 2]  +  A[1, 2]))}")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, matrix([2, 3]), 1)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, matrix([2, 3]), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, matrix([2, 3]), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, matrix([2, 3]), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, matrix([2, 3]), vector(3))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, matrix([2, 3]), matrix([2, 4]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, matrix([2, 3]), matrix([3, 3]))
        self.failUnlessEqual(str(2 * matrix([2, 3], "A")), "unnamed_matrix{=((A[0, 0] * unnamed_scalar{=2.0}, A[0, 1] * unnamed_scalar{=2.0}, A[0, 2] * unnamed_scalar{=2.0}), (A[1, 0] * unnamed_scalar{=2.0}, A[1, 1] * unnamed_scalar{=2.0}, A[1, 2] * unnamed_scalar{=2.0}))}")
        self.failUnlessEqual(str(matrix([2, 3], "A") * 2), "unnamed_matrix{=((A[0, 0] * unnamed_scalar{=2.0}, A[0, 1] * unnamed_scalar{=2.0}, A[0, 2] * unnamed_scalar{=2.0}), (A[1, 0] * unnamed_scalar{=2.0}, A[1, 1] * unnamed_scalar{=2.0}, A[1, 2] * unnamed_scalar{=2.0}))}")
        self.failUnlessEqual(str(matrix([2, 3], "A") * vector(3, "a")), "unnamed_vector{=(A[0, 0] * a[0]  +  A[0, 1] * a[1]  +  A[0, 2] * a[2], A[1, 0] * a[0]  +  A[1, 1] * a[1]  +  A[1, 2] * a[2])}")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__mul__, vector(2, "a"), matrix([2, 3], "A"))
        self.failUnlessEqual(str(matrix([2, 3], "A") * matrix([3, 2], "B")), "unnamed_matrix{=((A[0, 0] * B[0, 0]  +  A[0, 1] * B[1, 0]  +  A[0, 2] * B[2, 0], A[0, 0] * B[0, 1]  +  A[0, 1] * B[1, 1]  +  A[0, 2] * B[2, 1]), (A[1, 0] * B[0, 0]  +  A[1, 1] * B[1, 0]  +  A[1, 2] * B[2, 0], A[1, 0] * B[0, 1]  +  A[1, 1] * B[1, 1]  +  A[1, 2] * B[2, 1]))}")
        self.failUnlessEqual(str(matrix([2, 3], "A") / 2.0), "unnamed_matrix{=((unnamed_scalar{=0.5} * A[0, 0], unnamed_scalar{=0.5} * A[0, 1], unnamed_scalar{=0.5} * A[0, 2]), (unnamed_scalar{=0.5} * A[1, 0], unnamed_scalar{=0.5} * A[1, 1], unnamed_scalar{=0.5} * A[1, 2]))}")
        self.failUnlessEqual(str(matrix([2, 3], "A") / 2), "unnamed_matrix{=((unnamed_scalar{=0.0} * A[0, 0], unnamed_scalar{=0.0} * A[0, 1], unnamed_scalar{=0.0} * A[0, 2]), (unnamed_scalar{=0.0} * A[1, 0], unnamed_scalar{=0.0} * A[1, 1], unnamed_scalar{=0.0} * A[1, 2]))}")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, scalar(), matrix([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, scalar() + scalar(), matrix([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, vector(1), matrix([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, matrix([2, 3]), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, matrix([2, 3]), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, matrix([2, 3]), vector(1))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, matrix([2, 3]), matrix([2, 3]))


class TrafoTestCase(unittest.TestCase):

    def testInit(self):
        self.failUnlessEqual(str(trafo((2, 3))), "unnamed_trafo{=(matrix: unnamed_trafo_matrix{=((unnamed_trafo_matrix[0, 0], unnamed_trafo_matrix[0, 1], unnamed_trafo_matrix[0, 2]), (unnamed_trafo_matrix[1, 0], unnamed_trafo_matrix[1, 1], unnamed_trafo_matrix[1, 2]))}, vector: unnamed_trafo_vector{=(unnamed_trafo_vector[0], unnamed_trafo_vector[1])})}")
        self.failUnlessEqual(str(trafo((2, 3), "A")), "A{=(matrix: A_matrix{=((A_matrix[0, 0], A_matrix[0, 1], A_matrix[0, 2]), (A_matrix[1, 0], A_matrix[1, 1], A_matrix[1, 2]))}, vector: A_vector{=(A_vector[0], A_vector[1])})}")
        self.failUnlessEqual(str(trafo((matrix([2, 3], "B"), vector(2, "c")), "A")), "A{=(matrix: B{=((B[0, 0], B[0, 1], B[0, 2]), (B[1, 0], B[1, 1], B[1, 2]))}, vector: c{=(c[0], c[1])})}")
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), trafo, (matrix([2, 3]), vector(3)))

    def testAccess(self):
        m = matrix([2, 3])
        v = vector(2)
        t = trafo((m, v))
        self.failUnlessEqual(t.getmatrix(), m)
        self.failUnlessEqual(t.getvector(), v)

    def testMath(self):
        self.failUnlessEqual(str(-trafo([2, 3], "A")), str(trafo((-matrix([2, 3], "A_matrix"), -vector(2, "A_vector")))))
        self.failUnlessEqual(str(trafo([2, 3], "A") + trafo([2, 3], "B")), str(trafo((matrix([2, 3], "A_matrix") + matrix([2, 3], "B_matrix"), vector(2, "A_vector") + vector(2, "B_vector")))))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, 1, trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, trafo([2, 3]), 1)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, scalar(), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, trafo([2, 3]), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, scalar() + scalar(), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, trafo([2, 3]), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, vector(2), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, trafo([2, 3]), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, vector(3), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, trafo([2, 3]), vector(3))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, matrix([2, 3]), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, trafo([2, 3]), matrix([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, trafo([2, 3]), trafo([2, 4]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__add__, trafo([2, 3]), trafo([3, 3]))
        self.failUnlessEqual(str(trafo([2, 3], "A") - trafo([2, 3], "B")), str(trafo((matrix([2, 3], "A_matrix") - matrix([2, 3], "B_matrix"), vector(2, "A_vector") - vector(2, "B_vector")))))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, 1, trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, trafo([2, 3]), 1)
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, scalar(), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, trafo([2, 3]), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, scalar() + scalar(), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, trafo([2, 3]), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, vector(2), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, trafo([2, 3]), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, vector(3), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, trafo([2, 3]), vector(3))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, matrix([2, 3]), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, trafo([2, 3]), matrix([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, trafo([2, 3]), trafo([2, 4]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__sub__, trafo([2, 3]), trafo([3, 3]))
        self.failUnlessEqual(str(trafo([2, 3], "A") * trafo([3, 4], "B")), str(trafo((matrix([2, 3], "A_matrix") * matrix([3, 4], "B_matrix"), vector(2, "A_vector") + matrix([2, 3], "A_matrix") * vector(3, "B_vector")))))
        self.failUnlessEqual(str(2 * trafo([2, 3], "A")), str(trafo((2 * matrix([2, 3], "A_matrix"), 2 * vector(2, "A_vector")))))
        self.failUnlessEqual(str(trafo([2, 3], "A") * 2), str(trafo((matrix([2, 3], "A_matrix") * 2, vector(2, "A_vector") * 2))))
        self.failUnlessEqual(str(scalar() * trafo([2, 3], "A")), str(trafo((scalar() * matrix([2, 3], "A_matrix"), scalar() * vector(2, "A_vector")))))
        self.failUnlessEqual(str(trafo([2, 3], "A") * scalar()), str(trafo((matrix([2, 3], "A_matrix") * scalar(), vector(2, "A_vector") * scalar()))))
        self.failUnlessEqual(str((scalar() + scalar()) * trafo([2, 3], "A")), str(trafo(((scalar() + scalar()) * matrix([2, 3], "A_matrix"), (scalar() + scalar()) * vector(2, "A_vector")))))
        self.failUnlessEqual(str(trafo([2, 3], "A") * (scalar() + scalar())), str(trafo((matrix([2, 3], "A_matrix") * (scalar() + scalar()), vector(2, "A_vector") * (scalar() + scalar())))))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__mul__, vector(2), trafo([2, 3]))
        self.failUnlessEqual(str(trafo([2, 3], "A") * vector(3, "a")), str(matrix([2, 3], "A_matrix") * vector(3, "a") + vector(2, "A_vector")))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__mul__, trafo([2, 3]), vector(2))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__mul__, matrix([3, 2]), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__mul__, trafo([2, 3]), matrix([3, 2]))
        self.failUnlessEqual(str(trafo([2, 3], "A") / 2.0), str(trafo((matrix([2, 3], "A_matrix") / 2.0, vector(2, "A_vector") / 2.0))))
        self.failUnlessEqual(str(trafo([2, 3], "A") / 2), str(trafo((matrix([2, 3], "A_matrix") / 2, vector(2, "A_vector") / 2))))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, scalar(), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, scalar() + scalar(), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, vector(1), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, matrix([2, 3]), trafo([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, trafo([2, 3]), scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, trafo([2, 3]), scalar() + scalar())
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, trafo([2, 3]), vector(1))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, trafo([2, 3]), matrix([2, 3]))
        self.failUnlessRaises((RuntimeError, AttributeError, TypeError), operator.__div__, trafo([2, 3]), trafo([2, 3]))

class SolverTestCase(unittest.TestCase):

    def testScalars(self):
        solver = Solver()
        a = scalar()
        solver.eq(2 + a, 3)
        self.failUnlessAlmostEqual(float(a), 1)
        self.failUnlessRaises(RuntimeError, solver.eq, 2 + a, 3) # the solver is broken after that
        solver = Solver()
        a = scalar()
        b = scalar()
        solver.eq(2*a, b)
        solver.eq(2 + a + b, 5)
        self.failUnlessAlmostEqual(float(a), 1)
        self.failUnlessAlmostEqual(float(b), 2)
        a = scalar()
        b = scalar()
        solver.eq(a, b)
        self.failUnlessRaises(Exception, solver.eq, a, b)

    def testVector(self):
        solver = Solver()
        a = vector(2)
        solver.eq(vector([1, 2]) + a, vector([3, 3]))
        self.failUnlessAlmostEqual(float(a.x), 2)
        self.failUnlessAlmostEqual(float(a.y), 1)
        b = vector(2)
        solver.eq(a*b) # => solver.eq(a*b, 0)
        solver.eq(b.y, 2)
        self.failUnlessAlmostEqual(float(b.x), -1)

    def testMatrix(self):
        solver = Solver()
        A = matrix([2, 2])
        solver.eq(A*vector([1, -1]), vector([2, 0]))
        solver.eq(A*vector([1, 1]), vector([0, 2]))
        self.failUnlessAlmostEqual(float(A[0, 0]), 1)
        self.failUnlessAlmostEqual(float(A[0, 1]), -1)
        self.failUnlessAlmostEqual(float(A[1, 0]), 1)
        self.failUnlessAlmostEqual(float(A[1, 1]), 1)
        B = matrix([2, 2])
        solver.eq(A*B, identitymatrix(2))
        self.failUnlessAlmostEqual(float(B[0, 0]), 0.5)
        self.failUnlessAlmostEqual(float(B[0, 1]), 0.5)
        self.failUnlessAlmostEqual(float(B[1, 0]), -0.5)
        self.failUnlessAlmostEqual(float(B[1, 1]), 0.5)
        self.failUnlessAlmostEqual(float((B * vector([2, 0]))[0]), 1)
        self.failUnlessAlmostEqual(float((B * vector([2, 0]))[1]), -1)
        self.failUnlessAlmostEqual(float((B * vector([0, 2]))[0]), 1)
        self.failUnlessAlmostEqual(float((B * vector([0, 2]))[1]), 1)

    def testTrafo(self):
        solver = Solver()
        A = trafo([2, 2])
        solver.eq(A.getmatrix()[0, 0], A.getmatrix()[1, 1]) # restrict to rotation + scaling + translation
        solver.eq(A.getmatrix()[0, 1], -A.getmatrix()[1, 0])
        solver.eq(A*vector([0, 1]), vector([-1, 2]))
        solver.eq(A*vector([1, 0]), vector([1, 4]))
        self.failUnlessEqual(float((A * vector([0, 0]))[0]), 1)
        self.failUnlessEqual(float((A * vector([0, 0]))[1]), 2)
        self.failUnlessEqual(float(A.getmatrix()[0, 0]), 0)
        self.failUnlessEqual(float(A.getmatrix()[0, 1]), -2)
        self.failUnlessEqual(float(A.getmatrix()[1, 0]), 2)
        self.failUnlessEqual(float(A.getmatrix()[1, 1]), 0)
        B = trafo([2, 2])
        solver.eq(A*B, identitytrafo(2))
        self.failUnlessEqual(float((B * vector([0, 0]))[0]), -1)
        self.failUnlessEqual(float((B * vector([0, 0]))[1]), 0.5)
        self.failUnlessEqual(float(B.getmatrix()[0, 0]), 0)
        self.failUnlessEqual(float(B.getmatrix()[0, 1]), 0.5)
        self.failUnlessEqual(float(B.getmatrix()[1, 0]), -0.5)
        self.failUnlessEqual(float(B.getmatrix()[1, 1]), 0)


if __name__ == "__main__":
    unittest.main()
