// Created on: 1991-05-27
// Created by: Arnaud BOUZY
// Copyright (c) 1991-1999 Matra Datavision
// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include <Expr.hxx>
#include <Expr_ArgTanh.hxx>
#include <Expr_GeneralExpression.hxx>
#include <Expr_NamedUnknown.hxx>
#include <Expr_Operators.hxx>
#include <Expr_Square.hxx>
#include <Expr_Tanh.hxx>
#include <Standard_Type.hxx>
#include <TCollection_AsciiString.hxx>

IMPLEMENT_STANDARD_RTTIEXT(Expr_ArgTanh, Expr_UnaryExpression)

Expr_ArgTanh::Expr_ArgTanh(const Handle(Expr_GeneralExpression)& exp)
{
  CreateOperand(exp);
}

Handle(Expr_GeneralExpression) Expr_ArgTanh::ShallowSimplified() const
{
  Handle(Expr_GeneralExpression) op = Operand();
  if (op->IsKind(STANDARD_TYPE(Expr_NumericValue)))
  {
    Handle(Expr_NumericValue) valop = Handle(Expr_NumericValue)::DownCast(op);
    return new Expr_NumericValue(ATanh(valop->GetValue()));
  }
  if (op->IsKind(STANDARD_TYPE(Expr_Tanh)))
  {
    return op->SubExpression(1);
  }
  Handle(Expr_ArgTanh) me = this;
  return me;
}

Handle(Expr_GeneralExpression) Expr_ArgTanh::Copy() const
{
  return new Expr_ArgTanh(Expr::CopyShare(Operand()));
}

Standard_Boolean Expr_ArgTanh::IsIdentical(const Handle(Expr_GeneralExpression)& Other) const
{
  if (!Other->IsKind(STANDARD_TYPE(Expr_ArgTanh)))
  {
    return Standard_False;
  }
  Handle(Expr_GeneralExpression) op = Operand();
  return op->IsIdentical(Other->SubExpression(1));
}

Standard_Boolean Expr_ArgTanh::IsLinear() const
{
  if (ContainsUnknowns())
  {
    return Standard_False;
  }
  return Standard_True;
}

Handle(Expr_GeneralExpression) Expr_ArgTanh::Derivative(const Handle(Expr_NamedUnknown)& X) const
{
  if (!Contains(X))
  {
    return new Expr_NumericValue(0.0);
  }
  Handle(Expr_GeneralExpression) op    = Operand();
  Handle(Expr_GeneralExpression) derop = op->Derivative(X);

  Handle(Expr_Square) sq = new Expr_Square(Expr::CopyShare(op));

  // 1 - X2

  Handle(Expr_Difference) thedif = 1.0 - sq->ShallowSimplified();

  // ArgTanh'(F(X)) = F'(X)/(1 - F(X)2)
  Handle(Expr_Division) thediv = derop / thedif->ShallowSimplified();

  return thediv->ShallowSimplified();
}

Standard_Real Expr_ArgTanh::Evaluate(const Expr_Array1OfNamedUnknown& vars,
                                     const TColStd_Array1OfReal&      vals) const
{
  Standard_Real val = Operand()->Evaluate(vars, vals);
  return ::Log((1.0 + val) / (1.0 - val)) / 2.0;
}

TCollection_AsciiString Expr_ArgTanh::String() const
{
  TCollection_AsciiString str("ATanh(");
  str += Operand()->String();
  str += ")";
  return str;
}
